/* Copyright (C) 2011-2018 RealVNC Ltd.  All Rights Reserved.
 */

#ifndef __VNCDISCOVERYTYPES_H__
#define __VNCDISCOVERYTYPES_H__

/**
* \file vncdiscoverysdktypes.h
*
* This is the header file that defines types for the VNC Discoverer SDK. 
* You should normally include vncdiscoverysdk.h instead of including 
* this file directly.
*
*/

#include <vnccall.h>
#include <vncint.h>

#ifdef __cplusplus
extern "C"
{
#endif

/** The type for the errors used by the Discovery SDK. */
typedef vnc_int32_t VNCDiscoverySDKError;
/** The type for request IDs used by the Discovery SDK. */
typedef vnc_uint32_t VNCDiscoverySDKRequestId;
/** The type for the time-out used by the Discovery SDK. */
typedef vnc_int32_t VNCDiscoverySDKTimeoutMicroseconds;
/** The status that is reported during status updates. */
typedef vnc_int32_t VNCDiscoverySDKDiscovererStatus;


/** Typedef for the Discovery SDK Instance. */
typedef struct VNCDiscoverySDKImpl VNCDiscoverySDKInstance;
/** Typedef for the Discoverer. */
typedef struct VNCDiscoverySDKDiscovererImpl VNCDiscoverySDKDiscoverer;
/** Typedef for the Entity. */
typedef struct VNCDiscoverySDKEntityImpl VNCDiscoverySDKEntity;

/**
 * The generator structure is used for creating strings which contain all the
 * information that would be held within a structure.
 *
 * \see VNCDiscoverySDKUtilCreateGenerator
 * \see vncdiscoverysdkutils.h
 */
typedef struct VNCDiscoverySDKGenerator VNCDiscoverySDKGenerator;

/**
 * The parser structure is used for parsing strings which contain all the
 * information that would be held within a structure.
 *
 * \see VNCDiscoverySDKUtilCreateParser
 * \see vncdiscoverysdkutils.h
 */
typedef struct VNCDiscoverySDKParser VNCDiscoverySDKParser;

/** Typedef for the key-value string pair */
typedef struct 
{
  /** The key string. */
  char *key;
  /** The value string. */
  char *value;
} VNCDiscoverySDKKeyValPair;

/**
 * \brief Structure for holding the information of a device.
 *
 * This structure will always be owned by the Discovery SDK. It is used to
 * provide the client application with information about a device.
 *
 * \see VNCDiscoverySDKChooseDiscovererForDevice,
 * VNCDiscoverySDKCancelDeviceChoice, VNCDiscoverySDKOfferDeviceToDiscoverer
 */
typedef struct
{
  /**
   * \brief The size of the device structure.
   * 
   * This describes the size in bytes of the structure. It allows the structure
   * to be extended in the future without breaking compatibility.
   */
  vnc_uint32_t size;
  /**
   * \brief The device identifier.
   */
  vnc_uint32_t deviceIdentifier;
  /**
   * \brief The name that identifies the provider which manages the device.
   */
  const char* pDeviceProviderName;
  /**
   * \brief The path to the provider which manages the device.
   */
  const char* pDeviceProviderPath;
  /**
   * \brief The size of the device properties array.
   */
  size_t devicePropertiesSize;
  /**
   * \brief Array of properties of the device.
   *
   * \see VNCDiscoverySDKKeyValPair
   */
  VNCDiscoverySDKKeyValPair *pDeviceProperties;
} VNCDiscoverySDKDevice;

/**
 * \brief Specifies the levels of interest a Discoverer might have in a device.
 *
 * \see VNCDiscoverySDKChooseDiscovererForDevice
 */
typedef enum
{
  /**
   * \brief The Discoverer is sure it can make use of the device and would need
   * exclusive access to it.
   *
   * If the Discoverer receives the device it will be the only Discoverer to
   * receive it.
   */
  VNCDeviceInterestLevelSureExclusive = 10,
  /**
   * \brief The Discoverer is sure it can make use of the device and can share
   * the access to it.
   *
   * If the Discoverer receives the device it might be shared with other
   * Discoverers. Therefore it needs to avoid setting the device in modes that
   * might conflict with the other Discoverers.
   */
  VNCDeviceInterestLevelSureShared = 15,
  /**
   * \brief The Discoverer might be able to make use of the device and would
   * need exclusive access to it.
   *
   * If the Discoverer receives the device it will be the only Discoverer to
   * receive it.
   */
  VNCDeviceInterestLevelMaybeExclusive = 30,
  /**
   * \brief The Discoverer might be able to make use of the device and can
   * share the access to it.
   *
   * If the Discoverer receives the device it might be shared with other
   * Discoverers. Therefore it needs to avoid setting the device in modes that
   * might conflict with the other Discoverers.
   */
  VNCDeviceInterestLevelMaybeShared = 35,
  /**
   * \brief The Discoverer does not need the device.
   *
   * The device will not be offered to the Discoverer.
   */
  VNCDeviceInterestLevelNotNeeded = 100
} VNCDeviceInterestLevel;

/**
 * \brief Structure holding information about a discoverer that is requesting
 * access to a device.
 *
 * This structure will always be owned by the Discovery SDK. It is used to
 * provide the client application with information about why the discoverer
 * might need the device. The application can use this to check with the user
 * which discoverer it prefers.
 *
 * \see VNCDiscoverySDKChooseDiscovererForDevice
 */
typedef struct
{
  /**
   * \brief The size of the discoverer request structure.
   * 
   * This describes the size in bytes of the structure. It allows the structure
   * to be extended in the future without breaking compatibility.
   */
  vnc_uint32_t size;
  /**
   * \brief The Discoverer needing access.
   *
   * The client application can retrieve the name of the discoverer by using
   * VNCDiscoverySDKGetDiscovererType.
   */
  VNCDiscoverySDKDiscoverer *pDiscoverer;
  /**
   * \brief A description of what the Discoverer does.
   *
   * This can be used to inform the user on what the Discoverer does in
   * general, what sort of servers it discovers.
   *
   * Can be NULL.
   *
   * This string is in UTF-8. The Discoverer might support configuring the
   * locale through VNCDiscoverySDKDiscovererSetProperty.
   */
  char *pDiscovererDescription;
  /**
   * The level of interest of the discoverer.
   *
   * \see VNCDeviceInterestLevel
   */
  VNCDeviceInterestLevel interestLevel;
  /**
   * \brief A description of the request.
   *
   * This can be used to inform the user why the Discoverer requires access to
   * a given device.
   *
   * Can be NULL.
   *
   * This string is in UTF-8. The Discoverer might support configuring the
   * locale through VNCDiscoverySDKDiscovererSetProperty.
   */
  char *pRequestDescription;
} VNCDiscovererRequestingAccess;

/**
 * \brief Structure holding information about the protocol a discoverer must
 * use when offered a device.
 *
 * It is used to provide the discoverer with information about which protocol
 * to use when connecting to a device offered to it, as well as extra
 * parameters needed to configure the protocol correctly.
 *
 * \see VNCDiscoverySDKChooseDiscovererForDevice,
 * VNCDiscoverySDKOfferDeviceToDiscovererForProtocol
 */
typedef struct
{
  /**
   * The protocol to be used. It should be one of the values defined by the
   * chosen discoverer.
   *
   * This must be a UTF-8 encoded and NUL-terminated.
   *
   * It cannot be NULL.
   */
  char *protocolType;

  /**
   * The number of VNCDiscoverySDKKeyValPair elements in parameters.
   */
  vnc_uint32_t parameterSize;

  /**
   * Array of parameters to use with the chosen protocol.
   */
  VNCDiscoverySDKKeyValPair *parameters;
} VNCDiscovererOfferParameter;

#ifdef _WIN32
  #include <windows.h>
  /** The event handle that is given by the Discoverer, or Entities and is
   * monitored by the SDK. */
  typedef HANDLE VNCDiscovererEventHandle;
  /** The value for the event handle that indicates the handle is "inactive" */
  #define VNCDiscovererNoEventHandle ((HANDLE)-1)
#else
  /** The event handle that is given by the Discoverer, or Entities and is
   * monitored by the SDK. */
  typedef int VNCDiscovererEventHandle;
  /** The value for the event handle that indicates the handle is "inactive" */
  #define VNCDiscovererNoEventHandle -1
#endif

/**
 * \brief Defines severity levels for the SDK Logger.
 *
 * This enumeration defines different severities for logging. The Discoverer should
 * specify one of them to the SDK. The SDK will log as appropriate.
 */
typedef enum
{
  /** Log a critical error */
  VNCDiscovererLogSeverityCriticalError = 0,
  /** Log a less serious error. */
  VNCDiscovererLogSeverityError = 10,
  /** Log a warning. */
  VNCDiscovererLogSeverityWarning = 20,
  /** Log some information. */
  VNCDiscovererLogSeverityInfo = 30,
  /** Log some debug details. */
  VNCDiscovererLogSeverityDebug = 80,
  /** Log the program flow - function entry. For this log level there should be
   * a matching exit log. */
  VNCDiscovererLogSeverityDebugFlowEntry = 90,
  /** Log the program flow - function exit. This log should be used to match an
   * entry log. All the entry logs should have a matching exit log. */
  VNCDiscovererLogSeverityDebugFlowExit = 91,
} VNCDiscovererLogSeverity;

/** No error occurred. Operation completed successfully. */
#define VNCDiscoverySDKErrorNone 0
/** One of the parameters passed in the request is not valid. */
#define VNCDiscoverySDKErrorInvalidParameter 1
/** Can't load the library needed for the Discoverer. */
#define VNCDiscoverySDKErrorLibraryNotLoaded 2
/** Not enough memory to complete the request. */
#define VNCDiscoverySDKErrorOutOfMemory 3
/** The requested item can not be found. */
#define VNCDiscoverySDKErrorNotFound 4
/** An unhandled exception occurred in the SDK thread, or an unexpected event
 * occurred in the SDK main thread loop. */
#define VNCDiscoverySDKErrorBadEvent 5
/** The Discoverer has not been loaded. */
#define VNCDiscoverySDKErrorDiscovererNotLoaded 6
/** The item has not been initialized. */
#define VNCDiscoverySDKErrorNotInitialized 7
/** The Discoverer is stopping. */
#define VNCDiscoverySDKErrorStopping 8
/** The requested operation is not supported. */
#define VNCDiscoverySDKErrorNotSupported 9
/** An item already exists. */
#define VNCDiscoverySDKErrorAlreadyExists 10
/** Access to a resource is not allowed. */
#define VNCDiscoverySDKErrorAccessDenied 11
/** The request can't be completed because initialization failed. */
#define VNCDiscoverySDKErrorUnableToInitialize 12
/** The request has been canceled. This can happen because of a call to
 * VNCDiscoverySDKDiscovererStop. */
#define VNCDiscoverySDKErrorCancelled 13
/** The request could not be completed before the timer expiry. */
#define VNCDiscoverySDKErrorTimedOut 14
/** The request could not be completed because the remote side returned an
 * error. */
#define VNCDiscoverySDKFailed 15
/** The request to operate on an entity could not be completed because the
 * entity is not known. */
#define VNCDiscoverySDKErrorUnknownEntity 16
/** The request failed because the signature check failed. This can happen if
 * the request involves downloading an XML document from a server (for example
 * a MirrorLink server) and the Signature field in the document doesn't match
 * the document itself.
 */
#define VNCDiscoverySDKErrorSignatureValidationFailed 17
/** A license could not be added because it is not valid. */
#define VNCDiscoverySDKErrorLicenseNotValid 18
/** A feature of a discoverer or the SDK cannot be used because it is not
 * licensed. */
#define VNCDiscoverySDKErrorFeatureNotLicensed 19
/**
 * An operation could not complete because the SDK is already running. The SDK
 * is considered as running if any discoverers have been created.
 */
#define VNCDiscoverySDKErrorIllegalWhileRunning 20
/**
 * An operation could not complete because the SDK is not yet running. The SDK
 * is considered as running if any discoverers have been created.
 */
#define VNCDiscoverySDKErrorIllegalWhileNotRunning 21
/** The Remote Server could not complete the request. */
#define VNCDiscoverySDKErrorRemoteServerRefusedRequest 22

/** Start of range of third-party Discoverer-specific error codes. The range
 * from this value onwards is reserved for the use of third parties developing
 * Discoverers. It is intended for error conditions that are specific to
 * particular Discoverer implementations and that do not map closely to error
 * codes defined in the 0 to (VNCDiscoverySDKErrorVENDOR - 1) range. 
 */
#define VNCDiscoverySDKErrorVENDOR 0x10000

/** ID for an unmatched request. */
#define VNCDiscoverySDKNoRequestId 0

/** No timeout shall be enforced. */
#define VNCDiscoverySDKNoTimeout -1
/** The call has to be instant, so the callback should be called before the
 * request method completes. */
#define VNCDiscoverySDKInstantCall 0

/** The Discoverer status is not defined. Most likely because the Discoverer is
 * not started. */
#define VNCDiscoverySDKDiscovererStatusUndefined 0
/** The Discoverer is waiting for new devices to appear. */
#define VNCDiscoverySDKDiscovererStatusWaiting  1
/** The Discoverer is scanning one, or more, new devices. */
#define VNCDiscoverySDKDiscovererStatusScanning 2

#ifdef __cplusplus
}
#endif

#endif /* !defined(__VNCDISCOVERYTYPES_H__) */

